/** globals WPMLMetaBox, ate_jobs_sync, LoadEateWidget */

import {action, computed, createStore, thunk, actionOn} from 'easy-peasy';
import {callEndpoint} from "wpml-common-js-source/src/Ajax";
import {__ as R__, both, complement, concat, filter, find, includes, isEmpty, isNil, length, map, pick, pipe, prop, propEq, propOr, pathOr} from "ramda";
import {dispatch} from "../../../../../../../src/js/shared/event";
import {setAction} from "../../../../../../../src/js/shared/store";
import "headjs/dist/1.0.0/head.load";
import noticeManager from './model/NoticeManager';
import {doesNeedReview, JOB_STATUSES, ATE_JOBS_STATUSES, isInProgress} from "../../jobs/model/jobs";
import ATEAPI from "../API/ATEAPI";
import Request from "../../shared/Request";
import {createStatusIconUpdater, JobsSync} from "../JobsSync";
import {JobsRetry} from "../JobsRetry";

const twoMinutes = 2 * 60 * 1000;
const oneMinutes = 60 * 1000;
const thirtySeconds = 30 * 1000;
const tenSeconds = 10 * 1000;

export const initializeStore = ({
									jobsToSync,
									needsReviewCount,
									notices,
									endpoints,
									shouldTranslateEverything,
									isAutomaticTranslations,
									ateConsole,
									urls
								}) => {

	const lock = {
		acquire: async () => {
			const {data} = await callEndpoint(prop('syncLock', endpoints), {action: 'aquire'});

			return data.success && pathOr(false, ['data', 'result'], data);
		},
		release: async () => {
			await callEndpoint(prop('syncLock', endpoints), {action: 'release'});
		}
	};

	needsReviewCount = parseInt(needsReviewCount);
	if (needsReviewCount > 0) {
		notices = noticeManager.replaceNotice(notices, noticeManager.createJobsToReviewNotice(urls, needsReviewCount));
	}

	const model = {
		jobsToSync,
		totalJobs: computed(pipe(prop('jobsToSync'), length)),
		automaticJobsInProgress: computed(pipe(
			prop('jobsToSync'),
			filter(prop('automatic')),
			filter(pipe(prop('status'), includes(
				R__,
				[JOB_STATUSES.WAITING_FOR_TRANSLATOR, JOB_STATUSES.IN_PROGRESS, JOB_STATUSES.ATE_NEEDS_RETRY]
			))),
			length
		)),
		needsReviewCount,

		notices,
		endpoints,
		credits: null,
		fetchingCredits: false,
		shouldTranslateEverything: shouldTranslateEverything == '1',
		isAutomaticTranslations: isAutomaticTranslations == '1',
		notEnoughCredit: false,

		synchronizeJobs: thunk(async (actions, payload, {getState}) => {
			if (!await lock.acquire()) {
				return;
			}

			const syncAndDownload = async ( previousJobsInProgress = 0, retry = 0) => {
				if ( retry > 10 ) {
					await lock.release();
					return;
				}
				let jobsToSync = getState().jobsToSync;

				const statusIconUpdater = createStatusIconUpdater();
				const ateAPI = new ATEAPI(new Request());
				const syncJobs = JobsSync(statusIconUpdater, ateAPI, actions);

				jobsToSync = await syncJobs(jobsToSync);

				const isJobInProgress = both(
					isInProgress,
					complement(propEq('ateStatus', ATE_JOBS_STATUSES.NO_CREDIT))
				);

				const jobsInProgress = pipe(filter(isJobInProgress), length)(jobsToSync);
				if ( !! jobsInProgress ) {
					const newRetry = previousJobsInProgress === jobsInProgress ? retry + 1 : retry;
					setTimeout(() => syncAndDownload(jobsInProgress, newRetry), ( retry + 1 ) *tenSeconds);
				} else {
					await lock.release();
				}

				actions.fetchCredits();
			}

			const ateAPI = new ATEAPI(new Request());
			const statusIconUpdater = createStatusIconUpdater();
			const retryJobs = JobsRetry(statusIconUpdater, ateAPI);
			await retryJobs();

			await syncAndDownload();
		}),

		retryJobs: thunk(async (actions) => {
			const retryJobs = JobsRetry(createStatusIconUpdater(), new ATEAPI(new Request()), actions);
			return retryJobs();
		}),

		translateEverything: thunk(async (actions, key = '', {getState}) => {
			dispatch('translate-everything-running', true);

			const {data: {success, data}} = await callEndpoint(prop('translate-everything', endpoints), {key});

			if (!isEmpty(propOr([], 'createdJobs', data))) {
				let jobsToSync = getState().jobsToSync;
				jobsToSync = concat( jobsToSync, map(({jobId}) => ({
					jobId, ateJobId: undefined, rid: undefined, automatic: true, status: JOB_STATUSES.IN_PROGRESS, review_status: undefined
				}), data.createdJobs));
				actions.setJobsToSync(jobsToSync);
			}

			if (success && data.key) {
				actions.translateEverything(data.key);
			}

			if (success && !data.key) {
				actions.setShouldTranslateEverything(false);
				actions.fetchCredits();
				actions.synchronizeJobs();
				dispatch('translate-everything-running', false);
			}

			if (!success && data.key === 'in-use') {
				setTimeout(actions.translateEverything, twoMinutes);
			}

			if (!success && data.key === 'waiting') {
				setTimeout(actions.translateEverything, thirtySeconds);
			}
		}),

		fetchCredits: thunk(async (actions) => {
			actions.setFetchingCredits(true);
			const {data} = await callEndpoint(endpoints.getCredits, {});
			actions.setFetchingCredits(false);
			if (data.success) {
				actions.setCredits(data.data);
			}
		}),

		setNeedsReviewCount: action((state, needsReviewCount) => {
			state.needsReviewCount = needsReviewCount;

			state.notices = needsReviewCount > 0 ?
				noticeManager.replaceNotice(state.notices, noticeManager.createJobsToReviewNotice(urls, needsReviewCount)) :
				noticeManager.removeNotice(state.notices, 'needsReview');
		}),

		setJobsToSync: setAction('jobsToSync'),
		setCredits: setAction('credits'),
		setFetchingCredits: setAction('fetchingCredits'),
		setShouldTranslateEverything: setAction('shouldTranslateEverything'),
		setNotices: setAction('notices'),

		setNotEnoughCredit: thunk(async (actions, notEnough, {getState}) => {
			if (notEnough !== getState().notEnoughCredit) {
				const {data} = await callEndpoint(endpoints.getATEJobsToSync, {});
				ateConsole.job_list = data.data;
				ateConsole.onIssueSolvingClose = () => actions.afterIssueSolving();

				const notices = notEnough ?
					noticeManager.replaceNotice(getState().notices, noticeManager.createNoCreditNotice(ateConsole, urls)) :
					noticeManager.removeNotice(getState().notices, 'noCredit');

				actions.setNotices(notices);
				actions.setNotEnoughCreditAction(notEnough);
			}
		}),

		afterIssueSolving: thunk(async (actions, payload, {getState}) => {
			const notices = noticeManager.removeNotice(getState().notices, 'noCredit');
			actions.setNotices(notices);
			actions.setNotEnoughCreditAction(false);
			actions.refreshJobsStatus();
		}),

		setNotEnoughCreditAction: setAction('notEnoughCredit'),

		onJobsToSync: actionOn(
			actions => actions.setJobsToSync,
			(state, target) => {
				state.needsReviewCount = needsReviewCount + pipe(filter(doesNeedReview), length)(target.payload);

				if (state.needsReviewCount > 0) {
					state.notices = noticeManager.replaceNotice(state.notices, noticeManager.createJobsToReviewNotice(urls, state.needsReviewCount));
				}
			}
		),
	};

	return createStore(model);
};

export const createSharedStore = (() => {
	let store = undefined;

	return (params) => {
		if (isNil(store)) {
			store = initializeStore(params);

			store.getActions().fetchCredits();

			if (prop('shouldTranslateEverything', ate_jobs_sync)) {
				store.getActions().translateEverything();
			} else if (prop('isSyncRequired', ate_jobs_sync)) {
				store.getActions().synchronizeJobs();
			}

			document.addEventListener('WPMLTooltipCreditsFixClicked', () => {
				if (params.ateConsole !== '') {
					params.ateConsole.onIssueSolvingClose = () => store.getActions().afterIssueSolving();
					head.js(params.urls.ateConsole, () => LoadEateWidget(params.ateConsole));
				}
			});
		}

		return store;
	}
})();
