import React from 'react';
import 'react-dates/initialize';
import {DateRangePicker} from 'react-dates';
import connect from "../context/connect";
import {StringsConsumer} from "../context/StringsContext";
import {SettingsConsumer} from "../context/SettingsProvider";
import {PaginationConsumer} from "../context/PaginationProvider";
import {JobsConsumer} from "../context/JobsProvider";
import Languages from "./filters/Languages";
import MultiSelect from "./filters/MultiSelect";
import TranslatedBy from "./filters/TranslatedBy";
import Sorts from './filters/Sorts';
import {JOB_STATUSES} from "../model/jobs";

export class Filters extends React.Component {

	constructor(props) {
		super(props);

		this.defaultState = {
			sentCalendarFocused: null,
			id: "",
			title: "",
			languages: {
				source: "any",
				target: ""
			},
			translatedBy: {
				scope: "all",
				value: 0
			},
			batch_name: "",
			status: "",
			needs_update: false,
			sent: {
				calendarFocused: null,
				startDate: null,
				endDate: null
			},
			deadline: {
				calendarFocused: null,
				startDate: null,
				endDate: null
			},
			sorting: {
				first: {
					sort: '-',
					direction: "asc",
				},
				second: {
					sort: '-',
					direction: "asc",
				}
			}
		};

		this.state = this.defaultState;
	}

	setId(event) {
		const target = event.currentTarget;
		this.setState({id: target.value});
	}

	setTitle(event) {
		const target = event.currentTarget;
		this.setState({title: target.value});
	}

	setBatchName(event) {
		const target = event.currentTarget;
		this.setState({batch_name: target.value});
	}

	setLanguages(languages) {
		this.setState({languages});
	}

	setSorting(sorting) {
		this.setState({sorting});
	}

	setFirstSort(value) {
		const newState = {
			firstSort: value
		};

		// Reset the second sort condition if the first's column is modified
		if (this.state.firstSort.sort !== value.sort) {
			newState.secondSort = {
				sort: '-',
				direaction: 'asc'
			}
		}

		this.setState(newState);
	}

	setSecondSort(value) {
		this.setState({secondSort: value});
	}

	filterColumnsInSecondSort() {
		if (this.state.firstSort.sort !== '-') {
			const {[this.state.firstSort.sort]: discard, ...filteredSortable} = this.props.sortable;
			return filteredSortable;
		} else {
			return {};
		}
	}

	setTranslatedBy(translatedBy) {
		this.setState({translatedBy});
	}

	setStatusFilter(status) {
		const hasNeedsUpdate = status.split(',').map(status => parseInt(status)).includes(JOB_STATUSES.NEEDS_UPDATE);

		this.setState({
			status,
			needs_update: hasNeedsUpdate,
		});
	}

	setDates(calendar, {startDate, endDate}) {
		const state = this.state[calendar];

		state.startDate = startDate;
		state.endDate = endDate;

		this.setState(state);
	}

	setCalendarFocus(calendar, calendarFocused) {
		const state = this.state[calendar];
		state.calendarFocused = calendarFocused;

		this.setState(state);
	}

	reset(e) {
		e.persist();

		this.setState({
			...this.defaultState,
			sent: {
				...this.state.sent,
				startDate: null,
				endDate: null
			},
			deadline: {
				...this.state.deadline,
				startDate: null,
				endDate: null
			}
		}, () => this.submit(e));
	}

	submit(e) {
		e.preventDefault();

		let params = {};

		if (this.state.id !== "") {
			params.id = this.state.id;
		}
		if (this.state.title !== "") {
			params.title = this.state.title;
		}
		if (this.state.batch_name !== "") {
			params.batch_name = this.state.batch_name;
		}
		if (this.state.languages.source !== "any") {
			params.source_language = this.state.languages.source;
		}
		if (this.state.languages.target !== "") {
			params.target_language = this.state.languages.target;
		}

		if (this.state.translatedBy.scope !== "all") {
			params.scope = this.state.translatedBy.scope;
			if (this.state.translatedBy.value) {
				params.translated_by = this.state.translatedBy.value;
			}
		}

		if (this.state.status) {
			const status = this.state.status.split(',').filter(status => status != JOB_STATUSES.NEEDS_UPDATE).join(',');
			if (status) {
				params.status = status;
			}

			params.needs_update = this.state.needs_update ? 'include' : 'exclude';
		}

		if (this.state.sent.startDate) {
			params.sent_from = this.state.sent.startDate.format('YYYY-MM-DD');
		}
		if (this.state.sent.endDate) {
			params.sent_to = this.state.sent.endDate.format('YYYY-MM-DD');
		}
		if (this.state.deadline.startDate) {
			params.deadline_from = this.state.deadline.startDate.format('YYYY-MM-DD');
		}
		if (this.state.deadline.endDate) {
			params.deadline_to = this.state.deadline.endDate.format('YYYY-MM-DD');
		}

		if (this.state.sorting.first.sort !== '-') {
			let sorting = {};
			sorting[this.state.sorting.first.sort] = this.state.sorting.first.direction;

			if (this.state.sorting.second.sort !== '-') {
				sorting[this.state.sorting.second.sort] = this.state.sorting.second.direction;
			}

			params.sorting = sorting;
		}

		this.props.setPage(1);
		this.props.filterJobs(params);
	}

	render() {
		const strings = this.props.strings.filters;

		return (
			<form className="wpml-jobs-filters" onSubmit={e => this.submit(e)}>
				<span>
					<input
						name="id"
						type="search"
						placeholder={strings.id}
						value={this.state.id}
						onChange={e => this.setId(e)}
					/>
				</span>

				<span>
					<input
						name="title"
						type="search"
						placeholder={strings.title}
						value={this.state.title}
						onChange={e => this.setTitle(e)}
					/>
				</span>

				<span>
					<input
						name="batch_name"
						type="search"
						placeholder={strings.batch_name}
						value={this.state.batch_name}
						onChange={e => this.setBatchName(e)}
					/>
				</span>

				<Languages
					value={this.state.languages}
					onChange={languages => this.setLanguages(languages)}
				/>

				<TranslatedBy
					value={this.state.translatedBy}
					onChange={value => this.setTranslatedBy(value)}
				/>

				<MultiSelect
					id="status"
					label={strings.with}
					value={this.state.status}
					onChange={value => this.setStatusFilter(value)}
					options={this.props.jobsStatuses}
					placeholder={strings.selectStatus}
				/>

				<span>
					<label>
						{strings.sentBetween}
					</label>
					<DateRangePicker
						key="sent"
						startDateId="sent-start"
						endDateId="sent-end"
						startDate={this.state.sent.startDate}
						endDate={this.state.sent.endDate}
						onDatesChange={e => this.setDates("sent", e)}
						focusedInput={this.state.sent.calendarFocused}
						onFocusChange={focus => this.setCalendarFocus("sent", focus)}
						showClearDates={true}
						numberOfMonths={1}
						isOutsideRange={() => false}
						small={true}
					/>
				</span>

				<span>
					<label>
						{strings.deadline}
					</label>
					<DateRangePicker
						key="deadline"
						startDateId="deadline-start"
						endDateId="deadline-end"
						startDate={this.state.deadline.startDate}
						endDate={this.state.deadline.endDate}
						onDatesChange={e => this.setDates("deadline", e)}
						focusedInput={this.state.deadline.calendarFocused}
						onFocusChange={focus => this.setCalendarFocus("deadline", focus)}
						showClearDates={true}
						numberOfMonths={1}
						isOutsideRange={() => false}
						small={true}
					/>
				</span>

				<Sorts
					value={this.state.sorting}
					onChange={value => this.setSorting(value)}
				/>

				<input
					type="submit"
					className="button button-secondary"
					value={strings.filter}
				/>

				<input
					type="reset"
					className="button-reset"
					value={"× " + strings.reset}
					onClick={e => this.reset(e)}
				/>

			</form>
		);
	}
}

export default connect(Filters, StringsConsumer, SettingsConsumer, JobsConsumer, PaginationConsumer);
