import React, {Component} from 'react';

import JobAPI from "../api/JobsAPI";
import Request from "../../shared/Request";
import {SettingsConsumer} from "./SettingsProvider";
import {StringsConsumer} from "./StringsContext";
import connect from "./connect";

import {canBeCanceled, canBeBatchSynced} from "../model/jobs";
import {DownloadXLIFF} from "../model/DownloadXLIFF";
import {create as createSyncATEJobs} from "../model/SyncATEJobs";

const JobsContext = React.createContext({});

export const JOB_STATUSES = {
	CANCELLED: 0,
	WAITING_FOR_TRANSLATOR: 1,
	IN_PROGRESS: 2,
	TRANSLATION_READY_TO_DOWNLOAD: 4,
	COMPLETED: 10
};

export class JobsProvider extends Component {
	constructor(props) {
		super(props);

		this.state = {
			progress: false,
			confirmation: false,
			error: false,
			jobs: [],
			totalJobsCount: 0,
			filters: {}
		};

		this.jobAPI = this.props.jobAPI ? this.props.jobAPI : new JobAPI(new Request());

		this.getPageParams = (page) => {
			return {
				limit: this.props.pageSize,
				offset: (page - 1) * this.props.pageSize
			};
		}
	}

	handleError(error) {
		this.setState({
			error,
			progress: false,
		});
	}

	displayConfirmation(confirmation) {
		this.setState({confirmation}, () => {
			setTimeout(() => {
				this.setState({confirmation: false});
			}, 5000);
		});
	}

	componentDidMount() {
		this.loadJobs()
			.then(({jobs}) => {
				if (this.props.siteKey && this.props.translationService && this.props.translationService.id > 0) {
					this.jobAPI.synchronizeStatusesWithTP()
						.then(updatedStatuses => this.updateJobStatuses(updatedStatuses));
				}
				return jobs;
			})
			.then(jobs => {
				if (jobs) {
					this.setState(async ({jobs}) => {
						jobs = await createSyncATEJobs().sync(jobs);

						return {jobs};
					});
				}
				return jobs;
			})
			.catch((err) => {
				this.handleError(err);
			});
	}


	updateJobStatuses(updatedStatuses) {
		this.setState(state => {
			const jobs = state.jobs.map(job => {
				const newStatus = updatedStatuses.find(el => {
					return el.id === job.id && el.type === job.type;
				});

				if (newStatus) {
					job.status = newStatus.status;
				}

				return job;
			});

			return {
				...state,
				jobs
			}
		});
	}

	loadJobs(page = 1) {
		if (this.state.progress) {
			return;
		}

		this.setState({progress: {msg: this.props.strings.progressMessages.loadingJobs}});

		const params = {...this.state.filters, ...(this.getPageParams(page))};

		return this.jobAPI.loadJobs(params).then(result => {
			result.jobs = result.jobs.map(job => {
				if (job.ts_status) {
					job.ts_status = JSON.parse(job.ts_status);
				}

				return job;
			});

			return result;
		}).then((result) => {
			this.setState({
				progress: false,
				error: false,
				jobs: result.jobs,
				totalJobsCount: result.total
			});

			return result;
		});
	}

	filterJobs(filters = {}) {
		return new Promise((resolve, reject) => {
			this.setState({filters}, () => {
				resolve(this.loadJobs(1, filters));
			});
		});
	}

	applyTranslations(jobs) {
		if (this.state.progress) {
			return;
		}

		jobs = jobs.filter(job => job.status === JOB_STATUSES.TRANSLATION_READY_TO_DOWNLOAD || job.status === JOB_STATUSES.COMPLETED);
		if (jobs.length === 0) {
			return;
		}

		this.setState({
			progress: {
				msg: this.props.strings.progressMessages.applyingTranslations,
				jobs
			}
		});

		return this.jobAPI.applyTranslation(jobs).then((updatedStatusesOfJobs) => {
			return this.state.jobs.map((job) => {
				const status = updatedStatusesOfJobs.find(({id, type}) => id === job.id && type === job.type);

				if (status) {
					job.status = JOB_STATUSES.COMPLETED;
				}

				return job;
			});
		}).then((jobs) => {
			this.setState({
				jobs,
				progress: false
			});

			return jobs;
		}).then(jobs => {
			this.displayConfirmation(this.props.strings.confirmations.applyingTranslations);

			return jobs;
		}).catch(e => this.handleError(e));
	}

	cancelJobs(jobs) {
		if (this.state.progress) {
			return;
		}

		jobs = jobs.filter(canBeCanceled);
		if (jobs.length === 0) {
			return;
		}

		this.setState({
			progress: {
				msg: this.props.strings.progressMessages.cancelJobs,
				jobs
			}
		});

		return this.jobAPI.cancelJobs(jobs).then((canceledJobs) => {
			return this.state.jobs.map((job) => {
				const status = canceledJobs.find(({id, type}) => id === job.id && type === job.type);

				if (status) {
					job.status = JOB_STATUSES.CANCELLED;
				}

				return job;
			});
		}).then((jobs) => {
			this.setState({
				jobs,
				progress: false
			});

			return jobs;
		}).then(jobs => {
			this.displayConfirmation(this.props.strings.confirmations.cancelJobs);

			return jobs;
		}).catch(e => this.handleError(e));
	}

	syncBatch(job) {
		if (this.state.progress) {
			return;
		}

		if (!job.batch.tp_id) {
			return;
		}

		const jobs = this.state.jobs.filter(canBeBatchSynced);

		this.setState({
			progress: {
				msg: this.props.strings.progressMessages.syncBatch,
				jobs
			}
		});

		return this.jobAPI.syncBatch([job.batch.tp_id]).then(() => {
			const jobs = this.state.jobs.map(stateJob => {
				if (stateJob.batch.tp_id === job.batch.tp_id) {
					stateJob.isBatchSynced = true;
				}

				return stateJob;
			});

			this.setState({
				jobs,
				progress: false
			});

			return jobs;
		}).then(jobs => {
			this.displayConfirmation(this.props.strings.confirmations.syncBatch);

			return jobs;
		}).catch(e => this.handleError(e));
	}

	assignTranslator(job, translatorId) {
		if (this.state.progress) {
			return;
		}

		this.setState({
			progress: {
				msg: this.props.strings.progressMessages.assignTranslator,
				jobs: [job]
			}
		});

		return this.jobAPI.assignTranslator(job, translatorId).then(result => {
			if (!result.assigned) {
				throw new Error("A local translator could not be assigned.");
			}

			const translator = this.props.localTranslators.find(({value: id}) => id == translatorId);

			const jobs = this.state.jobs.map(stateJob => {
				if (job.id === stateJob.id && job.type === stateJob.type) {
					stateJob.translator_name = translator.label;
				}

				return stateJob;
			});

			this.setState({
				jobs,
				progress: false,
			});
		}).then(() => {
			this.displayConfirmation(this.props.strings.confirmations.assignTranslator);

			return job;
		}).catch(e => this.handleError(e));
	}

	async downloadXLIFF(job) {
		try {
			if (this.state.progress) {
				return;
			}

			this.setState({
				progress: {
					msg: this.props.strings.progressMessages.downloadingXliff,
					jobs: [job]
				}
			});

			const downloadXliff = new DownloadXLIFF(this.jobAPI);
			await downloadXliff.run(job);

			this.setState({progress: false});
			this.displayConfirmation(this.props.strings.confirmations.downloadingXliff);
		} catch (err) {
			this.handleError('Failed download XLIFF file. Please try later.');
		}
	}

	hasExternalActions() {
		return this.state.jobs.find(job => job.ts_status && job.ts_status.links.length) !== undefined;
	}

	render() {
		const {children} = this.props;

		return (
			<JobsContext.Provider
				value={{
					...this.state,
					applyTranslations: (jobs) => this.applyTranslations(jobs),
					loadJobs: (page = 1) => this.loadJobs(page).catch(this.handleError.bind(this)),
					filterJobs: (filers = {}) => this.filterJobs(filers),
					syncBatch: job => this.syncBatch(job),
					cancelJobs: jobs => this.cancelJobs(jobs),
					assignTranslator: (job, translatorId) => this.assignTranslator(job, translatorId),
					hasExternalActions: () => this.hasExternalActions(),
					downloadXLIFF: job => this.downloadXLIFF(job),
				}}
			>
				{children}
			</JobsContext.Provider>
		);
	}
}

export const JobsConsumer = JobsContext.Consumer;

export const ConnectedJobProvider = connect(JobsProvider, SettingsConsumer, StringsConsumer);
