import 'url-polyfill';
import MCSCustomFieldSettingsFilter from './MCSCustomFieldSettingsFilter.js';
import MCSCustomFieldSettingsNavigator from './MCSCustomFieldSettingsNavigator.js';

class MCSCustomFieldSettings {

	constructor(form, tools) {
		this.DEFAULT_PAGE_SIZE = 20;
		this.SCROLL_DURATION = 1200;
		this.MIN_SEARCH_LENGTH = 1;
		this.SEARCH_TIMEOUT_VALUE = 500;

		this.form = form;
		this.tools = tools;

		this.initData();

		this.customFieldsType = this.form.dataset.type;

		document.addEventListener( 'icl-save-form-' + this.form.id, () => {
			this.saveFormCallback();
		} );
	}

	initData() {
		this.filterContainer = this.form.querySelector('.wpml_tm_search_container');
		delete this.filter;
		if (this.filterContainer) {
			this.filter = new MCSCustomFieldSettingsFilter(this.filterContainer,
				{minSearchLength: this.MIN_SEARCH_LENGTH, searchTimeOutValue: this.SEARCH_TIMEOUT_VALUE});
		}

		this.navigatorContainer = this.form.querySelector('.wpml_tm_pagination_container');
		delete this.navigator;
		if (this.navigatorContainer) {
			this.navigator = new MCSCustomFieldSettingsNavigator(this.navigatorContainer, this.DEFAULT_PAGE_SIZE);
		}

		if (this.form) {
			this.errors = this.form.querySelector('.otgs-notice__errors');
		}

		this.values = this.getValues();
	}

	bindEvents() {
		// When `onUpdateRequired` is fired, each respective component will display their spinner, then
		// as updateCustomFields returns a promise, both component should be able to use that
		// `.finally(() => ... ` to stop displaying the respective spinner.

		if (this.filter) {
			this.filter.bindEvents();
			this.filter.onUpdateRequired = () => this.updateCustomFields();
		}

		if (this.navigator) {
			this.navigator.bindEvents();
			this.navigator.onUpdateRequired = () => this.updateCustomFields();
		}

		wpmlCustomEvent('icl-bind-locks', { 'detail': this.form.id } );
	};

	getValues() {
		return {
			'checked': [...this.form.querySelectorAll( 'input[type="radio"]' )].map( field => field.checked ),
			'locked': [...this.form.querySelectorAll( 'input[name*="cf_unlocked"]' )].map( field => field.value )
		};
	}

	saveFormCallback() {
		this.values = this.getValues();
		this.updateCustomFields();
	}

	updateError(text) {
		if (text) {
			this.errors.querySelector('p').innerHTML = text;
			this.tools.show(this.errors);
		} else {
			this.tools.hide(this.errors);
		}
	}

	updateCustomFields() {
		if (JSON.stringify(this.values) !== JSON.stringify(this.getValues())) {
			this.form.querySelector('input[type="submit"]').click();
			return;
		}

		const data = {
			action: 'wpml_update_mcs_cf',
			type: this.customFieldsType,
			nonce: this.form.querySelector( '#_icl_nonce' ).value,
			items_per_page: this.navigator.getPageSize(),
			search: this.filter.getSearchText(),
			paged: this.navigator.getCurrentPage(),
			show_system_fields: Boolean((new URL(document.location)).searchParams.get('show_system_fields'))
		};

		const encodedData = Object.keys(data)
			.map((key) => encodeURIComponent(key) + '=' + encodeURIComponent(data[key]))
			.join('&');

		this.updateError();

		return fetch(ajaxurl,
			{
				method: 'POST',
				body: encodedData,
				headers: {
					'Content-Type': 'application/x-www-form-urlencoded; charset=utf-8'
				},
				credentials: 'same-origin'
			})
			.then(response => {
				if (!response.ok) {
					throw new Error(response.statusText);
				}
				return response.json();
			})
			.then(response => {
				if (response.success) {
					this.form.querySelector('.wpml-flex-table-body').innerHTML = response.data.body;
					this.navigatorContainer.outerHTML = response.data.pagination;
				} else {
					throw new Error(response.data.message);
				}
			})
			.then(() => {
				this.initData();
				this.bindEvents();

				this.tools.scrollTo(
					this.form.parentElement, this.tools.SCROLL_TO_BOTTOM, this.SCROLL_DURATION
				);

				this.filter.onUpdated();
				this.navigator.onUpdated();
			})
			.catch((reason) => {
				this.updateError(reason);

				this.filter.onUpdated();
				this.navigator.onUpdated();

				return reason;
			});
	};
}

export default MCSCustomFieldSettings;
