import Tools from './Shared/Tools';

class MCSCustomFieldSettingsNavigator {
	constructor(navigatorContainer, default_page_size) {
		this.default_page_size = default_page_size;

		this.onUpdateRequired = null;

		this.tools = new Tools();

		this.navigatorContainer = navigatorContainer;

		this.initData();
	}

	initData() {
		this.pageSize = this.default_page_size;
		this.currentPage = 1;

		this.pagesContainer = this.navigatorContainer.querySelector('.tablenav-pages');
		this.pageNumbers = this.pagesContainer.querySelectorAll('.page-numbers');
		this.displayLess = this.pagesContainer.querySelector('.display-less');
		this.displayAll = this.pagesContainer.querySelector('.display-all');
		this.next = this.pagesContainer.querySelector('.next');
		this.previous = this.pagesContainer.querySelector('.prev');
		this.spinner = this.navigatorContainer.querySelector('.spinner');

		this.pageSizeSelect = this.navigatorContainer.querySelector('select[name="wpml_tm_per_page"]');
		this.currentPageButton = this.navigatorContainer.querySelector('.current');

		this.customFieldsType = this.navigatorContainer.dataset.type;

		if (this.pageSizeSelect) {
			this.setPageSize(this.pageSizeSelect.value);
		}

		if (this.currentPageButton) {
			this.setCurrentPage(this.currentPageButton.innerText);
		}
	}

	bindEvents() {
		if (this.previous) {
			this.previous.onclick = (e) => this.onPreviousClick(e);
		}

		if (this.next) {
			this.next.onclick = (e) => this.onNextClick(e);
		}

		if (this.displayAll) {
			this.displayAll.onclick = (e) => this.onDisplayAllClick(e);
		}

		if (this.displayLess) {
			this.displayLess.onclick = (e) => this.onDisplayLessClick(e);
		}

		if (this.pageNumbers) {
			for (let i = 0; i < this.pageNumbers.length; i++) {
				let el = this.pageNumbers[i];
				let classList = el.classList;
				if (!classList.contains('dots') && !classList.contains('current') &&
					!classList.contains('prev') && !classList.contains('next')) {
					el.onclick = (e) => {
						e.preventDefault();
						this.setCurrentPage(e.currentTarget.innerText);
						this.updateNavigator();
					};
				}
			}
		}

		if (this.pageSizeSelect) {
			this.pageSizeSelect.onchange = () => this.onPageSizeChange();
		}
	}

	onPreviousClick(e) {
		e.preventDefault();
		this.goToPrevious();
		this.updateNavigator();
	}

	onNextClick(e) {
		e.preventDefault();
		this.goToNext();
		this.updateNavigator();
	}

	onDisplayAllClick(e) {
		e.preventDefault();
		this.setCurrentPage(-1);
		this.updateNavigator();
	}

	onDisplayLessClick(e) {
		e.preventDefault();
		this.setCurrentPage(1);
		this.setPageSize(this.default_page_size);
		this.updateNavigator();
	}

	onPageSizeChange() {
		this.setPageSize(this.pageSizeSelect.value);
		this.updateNavigator();
	}


	getCurrentPage() {
		return this.currentPage;
	}

	setCurrentPage(value) {
		this.currentPage = MCSCustomFieldSettingsNavigator.getInt(value);
	}

	goToPrevious() {
		this.currentPage--;
	}

	goToNext() {
		this.currentPage++;
	}

	getPageSize() {
		return this.pageSize;
	}

	setPageSize(value) {
		this.pageSize = MCSCustomFieldSettingsNavigator.getInt(value);
	}

	updateNavigator() {
		this.tools.showSpinner(this.spinner);
		this.onUpdateRequired();
	}

	onUpdated() {
		this.tools.hideSpinner(this.spinner);
		this.initData();
		this.bindEvents();
	}

	static getInt( value ) {
		if (Number.isInteger(value)) {
			return value;
		}

		return parseInt(value.trim().replace(/\D+/g,''));
	}
}

export default MCSCustomFieldSettingsNavigator;