import {SyncJobs} from "./SyncJobs";
import {DownloadTranslations} from "./DownloadTranslations";
import {BatchSync} from "../../../jobs/model/BatchSync";
import {isJobEqual} from "../../../jobs/model/jobs";
import Request from "../../../shared/Request";
import JobsAPI from "../../../jobs/api/JobsAPI";

export class Process {
	/**
	 *
	 * @param {SyncJobs} syncJobs
	 * @param {DownloadTranslations} downloadTranslations
	 * @param {BatchSync} batchSync
	 * @param {Object} hooks
	 */
	constructor(syncJobs, downloadTranslations, batchSync, hooks) {
		this.syncJobs = syncJobs;
		this.downloadTranslations = downloadTranslations;
		this.batchSync = batchSync;
		this.hooks = hooks;
	}

	/**
	 *
	 * @param {Array<Object>}jobs
	 * @returns {Promise<boolean>}
	 */
	async run(jobs) {
		const {ready: jobsReadyToDownload, cancelled} = await this.syncJobs.sync(jobs);
		this.hooks.doAction('wpml-tm-job-translations-processed', cancelled);

		const asyncActions = [this.downloadTranslations.download(jobsReadyToDownload)];

		if (jobs.length > jobsReadyToDownload.length + cancelled.length) {
			const remainingJobs = jobs.filter(job => jobsReadyToDownload.find(syncedJob => isJobEqual(job, syncedJob)) === undefined);
			asyncActions.push(this.syncBatch(remainingJobs));
		}

		await Promise.all(asyncActions);

		return true;
	}


	async syncBatch(jobs) {
		const tpBatchIds = [...new Set(jobs.map(job => job.batch ? job.batch.tp_id : 0).filter(tpBatchId => !!tpBatchId))];
		if (!tpBatchIds) {
			return true;
		}

		await this.batchSync.run(tpBatchIds);

		const {ready: jobsReadyToDownload, cancelled} = await this.syncJobs.sync([]);
		this.hooks.doAction('wpml-tm-job-translations-processed', cancelled);
		return this.downloadTranslations.download(jobsReadyToDownload);
	}
}

/**
 *
 * @param hooks
 * @returns {Process}
 */
export const create = hooks => {
	const request = new Request();
	const jobsAPI = new JobsAPI(request);
	const batchSyncStatusCheckInterval = 10000;

	return new Process(
		new SyncJobs(jobsAPI),
		new DownloadTranslations(jobsAPI, hooks),
		new BatchSync(jobsAPI, hooks, batchSyncStatusCheckInterval),
		hooks
	);
};