/** globals  **/

import React from 'react';
import * as R from 'ramda';
import {useStoreState} from "easy-peasy";
import {JOB_STATUSES, ATE_JOBS_STATUSES, doesNeedReview} from "../jobs/model/jobs";
import WPMLTMSettings from '../shared/WPMLTMSettings';
import Request from "../shared/Request";
import WPMLAPI from "../ate/API/WPMLAPI";
import ATEAPI from "../ate/API/ATEAPI";
import ATEJobs from "../ate/ATEJobs";
import {__} from "wpml-common-js-source/src/i18n";
import sanitizeHtml from "sanitize-html";

export default ({jobData}) => {
    jobData = R.evolve({
        job_id: parseInt,
        status: parseInt,
        translator_id: parseInt
    }, jobData);

    const jobsToSync = useStoreState(R.prop('jobsToSync'));
    const ateJob = R.find(R.propEq('jobId', jobData.job_id), jobsToSync);
    if (ateJob) {
        jobData = {...jobData, ...ateJob, isATEJob: true};
    }

    if (shouldHideButtons(jobData)) {
        return <React.Fragment></React.Fragment>;
    }

    if (doesNeedReview(jobData)) {
        return <a className="button-secondary js-translation-queue-edit" href={R.propOr('', 'viewLink', jobData)}>
            {__('Review', 'wpml-translation-management')}
        </a>
    }

    return <React.Fragment>
        <a className="button-secondary js-translation-queue-edit" onClick={() => openATEEditor(jobData)} >
            {getButtonText(jobData)}
        </a>

        {
            isOwnerOfJob(jobData) && jobData.status !== JOB_STATUSES.COMPLETED && <ResignLink jobData={jobData}/>
        }

        {
            jobData.viewLink && <ViewLink jobData={jobData} />
        }
    </React.Fragment>
};

const ResignLink = ({jobData}) => {
    const onResign = e => {
        if (!confirm(__('Are you sure you want to resign from this job?', 'wpml-translation-management'))) {
            e.preventDefault()
        }
    };

    return <React.Fragment>
        <br/>
        <a className="link-resign"
           href={jobData.resignUrl}
           onClick={onResign}
        >
            { __( 'Resign', 'wpml-translation-management' ) }
        </a>
        </React.Fragment>
};

const ViewLink = ({jobData}) => <React.Fragment>
        <br/>
        {jobData.viewLink.startsWith('<a')
            ? <span dangerouslySetInnerHTML={{__html: sanitizeHtml(jobData.viewLink)}}/>
            : <span><a href={jobData.viewLink}>{__('View', 'wpml-translation-management')}</a></span>
        }
    </React.Fragment>;

const openATEEditor = jobData => {
    const request = new Request();

    const dependencies = {
        wpmlAPI: new WPMLAPI(request),
        ateAPI: new ATEAPI(request),
    };

    const ateJobs = new ATEJobs(dependencies);
    const settings = new WPMLTMSettings();

    ateJobs.assignJob({
        jobId: jobData.job_id,
        translatorId: settings.currentUser.ID,
    }).then(({assigned}) => {
        if (assigned) {
            window.location.href = jobData.edit_url;
        }
        if (!assigned) {
            console.log('Translation job not assigned');
        }
    });
};

const isOwnerOfJob = jobData => {
    const settings = new WPMLTMSettings();

    return settings.currentUser.ID === jobData.translator_id;
};

const shouldHideButtons = (jobData) => {
    const inProgressStatuses = [JOB_STATUSES.WAITING_FOR_TRANSLATOR, JOB_STATUSES.IN_PROGRESS, JOB_STATUSES.ATE_NEEDS_RETRY];

    return jobData.status === JOB_STATUSES.CANCELLED || jobData.isATEJob && (
        R.propEq('ateStatus', ATE_JOBS_STATUSES.NO_CREDIT, jobData) ||
        (jobData.automatic && R.includes(jobData.status, inProgressStatuses))
    );
};

const getButtonText = (jobData) => {
    if (jobData.status === JOB_STATUSES.COMPLETED) {
        return __('Edit', 'wpml-translation-management');
    } else if (isOwnerOfJob(jobData)) {
        return __('Translate', 'wpml-translation-management');
    } else {
        return __('Take and translate', 'wpml-translation-management');
    }
};
