import React, {useState} from 'react';
import {__} from 'wpml-common-js-source/src/i18n';
import {always, path, prop, pipe, ifElse, identity, filter, map} from 'ramda'
import Table from 'antd/lib/table';
import Divider from 'antd/lib/divider';
import Image from 'antd/lib/image';
import Button from "../../../../../../../src/js/shared/components/button";
import 'antd/lib/table/style/index.css';
import 'antd/lib/pagination/style/index.css';
import 'antd/lib/divider/style/index.css';
import 'antd/lib/image/style/index.css';
import SearchForm from './SearchForm';
import {useStoreActions, useStoreState} from "easy-peasy";
import Stars from "./Stars";


const matchPhrase = phrase => service => service.name.toLowerCase().indexOf(phrase.toLowerCase()) !== -1 || service.description.toLowerCase().indexOf(phrase.toLowerCase()) !== -1;

const highlightMatch = phrase => service => {
    const regexp = new RegExp(phrase, 'ig');
    return {...service, description: service.description.replace(regexp, '<b>$&</b>')};
};

const appendKey = service =>  ({...service, key: service.id});

const processServices = phrase => {
    if (phrase.length >= 3) {
        return pipe(filter(matchPhrase(phrase)), map(highlightMatch(phrase)), map(appendKey))
    } else {
        return map(appendKey);
    }
};

const NameColumn = ({service}) => {
    const logoPlaceholder = useStoreState(prop('logoPlaceholder'));
    const {logo_url, name, doc_url, url} = service;

    return <>
        <a target="_blank" rel="noopener" href={url}><Image alt={name} fallback={logoPlaceholder} src={logo_url} preview={false}/></a>
        {doc_url &&
        <a target="_blank" rel="noopener" href={doc_url}>{__('Documentation', 'wpml-translation-management')}</a>}
    </>
};

const ActionColumn = ({service}) => {
    const {id, active} = service;
    const selectService = useStoreActions(prop('selectService'));
    const [loading, setLoading] = useState(false);

    const onClick = () => {
        setLoading(true);

        selectService(service).finally(() => setLoading(false));
    };

    return <Button type='primary' disabled={active === 'active'}  className={'wpml-activate-service-' + id} onClick={onClick} loading={loading} >
        {__('Activate', 'wpml-translation-management')}
    </Button>;
};

export default ({sectionIndex, showPopularity, pagination}) => {
    const [search, setSearch] = useState('');

    let services = useStoreState(path(['sections', sectionIndex, 'services']));
    services = processServices(search)(services);

    let columns = [
        {
            title: __('Name', 'wpml-translation-management'),
            render: (text, service) => <NameColumn service={service}/>,
            sorter: (s1, s2) => s1.name > s2.name
        },
        {
            title: __('Description', 'wpml-translation-management'),
            dataIndex: 'description',
            render: (description) => <p dangerouslySetInnerHTML={{__html: description}} ></p>
        }
    ];
    if (showPopularity) {
        columns.push({
            title: __('Ranking', 'wpml-translation-management'),
	        width: 120,
            render: (text, service) => <Stars maxValue={5} value={service.rankings.primary}/>,
            sorter: (s1, s2) => s1.rankings.primary - s2.rankings.primary
        });
    }
    columns.push({
        title: '',
        render: (text, service) => <ActionColumn service={service}/>
    });

    return <div>

        <div className="tablenav top clearfix">
            <SearchForm setSearch={setSearch} />
        </div>

        <Divider/>

        <Table columns={columns} dataSource={services} pagination={pagination ? 'bottomRight' : false}/>

    </div>
};
