/*globals ajaxurl */
class WordCountDialog {

	/**
	 * @param {jQuery} $
	 */
	constructor( $ ) {
		this.$ = $;
		this.status = null;
		this.timeout = null;
	}

	init() {
		this.openLink = this.$('.js-word-count-dialog-open');

		this.openLink.click( ( e ) => {
			e.preventDefault();
			this.openDialog()
		});

		if ( '#words-count' === window.location.hash ) {
			this.openDialog();
		}
	}

	openDialog() {
		this.dialogInit();
		this.dialog.dialog({
			dialogClass  : 'word-count-dialog wpml-dialog otgs-ui-dialog',
			resizable    : false,
			modal        : true,
			width        : 450,
			autoResize   : true,
			title        : this.openLink.data('dialog-title'),
			closeOnEscape: true,
			buttons: [
				{
					text: this.openLink.data('cancel'),
					class: "alignleft button-secondary",
					click: e => this.cancelCount( e )
				}
			],
			open: () => this.repositionDialog(),
			close: () => this.restoreLegacyUIStyle()
		});
	}

	dialogInit() {
		this.$('#jquery-ui-style-css').prop('disabled', true); // prevent styling issues with 3rd part code

		if ( ! this.dialog ) {
			const loadingText = this.openLink.data('loading-text');
			this.dialog = this.$( `<div class="js-word-count-dialog"><span class="spinner is-active"></span>${loadingText}</div>` );
			this.fetchReportView();
			this.initDialogEvents();
			this.$( window ).resize( () => this.repositionDialog() );
		}
	}

	restoreLegacyUIStyle() {
		this.$('#jquery-ui-style-css').prop('disabled', false);
	}

	setDialogContent( report ) {
		this.$('.js-word-count-dialog').html( report );
	}

	setStatus( status ) {
		this.status = status;
		clearTimeout( this.timeout );

		if ( status === 'in-progress' ) {
			this.dialog.find( '.requested-type, .start-count' ).prop( 'disabled', true );
			this.dialog.find( '.spinner' ).addClass( 'is-active' );
			this.timeout = setTimeout( () => this.fetchReportView(), 5000 );
		} else if ( status === 'completed' ) {
			this.dialog.find( '.requested-type, .start-count' ).prop( 'disabled', false );
			this.dialog.find( '.spinner' ).removeClass( 'is-active' );
		}
	}

	fetchReportView() {
		const data = {
			action: 'wpml_word_count_get_report'
		};

		this.sendAjax( data ).then( data => {
			if ( this.status !== 'completed' ) {
				this.setStatus( data.status );
				this.setDialogContent( data.report );
				this.repositionDialog();
			}
		});
	}

	repositionDialog() {
		this.$(".otgs-ui-dialog .ui-dialog-content").css({
			"max-height": this.$( window ).height() - 180
		});

		this.$(".otgs-ui-dialog").css({
			"max-width": "95%"
		});

		this.dialog.dialog("option", "position", {
			my: "center",
			at: "center",
			of: window
		});
	}

	initDialogEvents() {
		this.dialog.on( 'click', '.start-count', e => this.startCount( e ) );
		this.dialog.on( 'click', '.cancel-count', e => this.cancelCount( e ) );
		this.dialog.on( 'change', '.requested-type', () => this.selectedTypesChanged() );
	}

	startCount( e ) {
		e.preventDefault();
		this.setStatus( 'in-progress' );

		const requestedTypes = {
			post_types: [],
			package_kinds: []
		};

		this.dialog.find( '.requested-type:checked' ).each( ( i, el ) => {
			const input = this.$( el );
			requestedTypes[ input.data( 'group' ) ].push( input.data( 'type' ) );
			const row = input.closest( 'tr' );
			row.removeClass( 'pending' ).removeClass( 'completed' ).addClass( 'in-progress' );
			row.find( '.js-row-count-words, .js-row-completed-items' ).text( 0 );
		});

		this.dialog.find( '.js-total-completed-items' ).text( '-' );
		this.dialog.find( '.js-total-count-words' ).text( '-' );

		const data = {
			action: 'wpml_word_count_start_count',
			requested_types: requestedTypes
		};

		this.sendAjax( data ).then( () => this.fetchReportView() );
	}

	cancelCount( e ) {
		e.preventDefault();
		this.setStatus( 'completed' );

		const data = {
			action: 'wpml_word_count_cancel_count'
		};

		this.sendAjax( data );
		this.dialog.dialog( 'close' );
	}

	selectedTypesChanged() {
		const selectedTypes = this.dialog.find( '.requested-type:checked' );
		const startCount = this.dialog.find( '.start-count' );

		if ( selectedTypes.length ) {
			startCount.prop( 'disabled', false );
		} else {
			startCount.prop( 'disabled', true );
		}
	}

	sendAjax( data ) {
		data.nonce = this.openLink.data('nonce');
		data.module = 'wpml_word_count';

		return this.$.post( ajaxurl, data ).then( ( response ) => {
			if ( response.success ) {
				return response.data;
			}
		});
	}
}

export default WordCountDialog;