/** globals WPMLMetaBox, ate_jobs_sync */

import {JobSyncStatusUpdate} from "./status-icons/PostListOrTMDashboard";
import {TranslationQueue} from "./status-icons/TranslationQueue";
import * as R from "ramda";
import {ATE_JOBS_STATUSES, JOB_STATUSES, isInProgress} from "../jobs/model/jobs";
import {dispatch} from "wpml-common-js-source/src/event";

const hasInsufficientBalance = R.propEq('ateStatus', ATE_JOBS_STATUSES.NO_CREDIT);
const isCancelled = R.pipe(R.prop('ateStatus'), R.includes(R.__, [ATE_JOBS_STATUSES.CANCELLED, ATE_JOBS_STATUSES.HIDDEN]));
const isCompletedInATE = R.pipe(R.prop('ateStatus'), R.includes(R.__, [ATE_JOBS_STATUSES.TRANSLATED, ATE_JOBS_STATUSES.DELIVERING, ATE_JOBS_STATUSES.EDITED]));
const isStillInProgress = R.complement(R.anyPass([hasInsufficientBalance, isCancelled, isCompletedInATE]));

export const JobsSync = function (statusIconUpdater, ateAPI, actions) {

    const appendATEStatus = (jobsToSync, syncedJobs) => {
        jobsToSync = R.map(job => {
            const syncedJob = R.find(R.propEq('ateJobId', job.ateJobId), syncedJobs);
            if (syncedJob) {
                return {...job, ateStatus: syncedJob.status, url: syncedJob.url};
            }

            return job;
        }, jobsToSync);

        const findEditedJobs = R.reject(syncedJob => R.find(R.propEq('ateJobId', syncedJob.ateJobId), jobsToSync));
        const buildEditedJob = syncedJob => ({
            jobId: syncedJob.wpmlJobId,
            ateJobId: syncedJob.ateJobId,
            url: syncedJob.url,
            status: JOB_STATUSES.COMPLETED,
            ateStatus: syncedJob.status,
            edited: true
        });
        const editedJobs = R.pipe(findEditedJobs, R.map(buildEditedJob))(syncedJobs);

        return R.concat(jobsToSync, editedJobs);
    };

    const updateStatusesOfSynchronizedJobs = (jobsToSync) => {
        const [insufficientBalanceJobs, other] = R.partition(hasInsufficientBalance(), jobsToSync);
        if (insufficientBalanceJobs.length > 0) {
            statusIconUpdater.displayInsufficientBalanceWarning(insufficientBalanceJobs);
            actions.setNotEnoughCredit(true);
        }

        const cancelledJobs = R.filter(isCancelled, other);
        statusIconUpdater.markJobsAsCancelled(cancelledJobs);
    };

    const synchronizeJobs = async (jobsToSync) => {
        let response = {};

        do {
            response = await ateAPI.sync(R.pick(['lockKey', 'ateToken', 'nextPage', 'numberOfPages'], response));
            if (response.jobs.length) {
                jobsToSync = appendATEStatus(jobsToSync, response.jobs);
                actions.setJobsToSync(jobsToSync);

                updateStatusesOfSynchronizedJobs(jobsToSync);
            }
        } while (response.ateToken);

        statusIconUpdater.revertBack(R.filter(isStillInProgress, jobsToSync));
        statusIconUpdater.setStatusIconToSpin(R.filter(R.propOr(false, 'edited'), jobsToSync));

        return jobsToSync;
    };

    const downloadJobs = async (jobsToSync) => {
        const isNotCompletedInWPML = R.complement(R.both(
            R.propEq('status', JOB_STATUSES.COMPLETED),
            R.complement(R.propOr(false, 'edited'))
        ));

		const completed = R.filter(R.both(isCompletedInATE, isNotCompletedInWPML), jobsToSync);
        const completedChunks = R.splitEvery(5, completed);

        const downloadChunk = async jobs => {
            const toInt = R.unary(parseInt);
            let downloadedJobs = await ateAPI.download(jobs, ate_jobs_sync.urls.currentUrl);
            downloadedJobs = R.map(R.evolve({jobId: toInt, status: toInt}), downloadedJobs);

            jobsToSync = jobsToSync.map(job => {
                const downloadedJob = downloadedJobs.find(R.propEq('jobId', job.jobId));
                if (downloadedJob) {
                    return {...job, ...downloadedJob, edited: false};
                }

                return job;
            });

            statusIconUpdater.updateDownloadedJobIcon(downloadedJobs);
            actions.setJobsToSync(jobsToSync);
        }

        // send max 10 download requests in parallel
        const requests = R.splitEvery(10, completedChunks);
        for (let index in requests) {
            await Promise.all(R.map(downloadChunk, requests[index]));
        }

        if (requests.length) {
            dispatch('jobs-downloaded', requests);
        }

        return jobsToSync;
    };

    return async (jobsToSync) => {
        statusIconUpdater.setStatusIconToSpin(R.filter(isInProgress, jobsToSync));
        jobsToSync = await synchronizeJobs(jobsToSync);
		jobsToSync = await downloadJobs(jobsToSync);
        statusIconUpdater.revertBack(R.filter(isStillInProgress, jobsToSync));

        return jobsToSync;
    };
};


export const createStatusIconUpdater = () => {
    const isTranslationQueue = !!document.getElementById('icl-translation-jobs');

    if (isTranslationQueue) {
        return new TranslationQueue(document, ate_jobs_sync.strings);
    } else {
        return new JobSyncStatusUpdate(
            document,
            ate_jobs_sync.strings,
            ate_jobs_sync.isTranslationManager,
            ate_jobs_sync.isAutomaticTranslations
        );
    }
};
