import * as R from "ramda";
import {doesNeedReview} from "../../jobs/model/jobs";
import initTooltip from "../../jobs/util/tooltip";
import {dispatch} from "wpml-common-js-source/src/event";
import {evolve, pluck, prop, propEq} from "ramda";

const disableLink = e => {
	e.preventDefault();
	return false;
};

export class JobSyncStatusUpdate {

	/**
	 *
	 * @param document
	 */
	constructor(document, strings, isTranslationManager, isAutomaticTranslations) {
		this.document = document;
		this.strings = strings;
		this.isTranslationManager = isTranslationManager;
		this.isAutomaticTranslations = isAutomaticTranslations;
		this.previousStates = [];
		this.getIcon = this.getIcon.bind(this);
	}

	notEnoughCreditToolTip(icon) {
		icon.style.color = this.isTranslationManager ? '#333' : '#c1c1c1';
		icon.style.cursor = this.isTranslationManager ? 'pointer' : 'default';
		const tmEditorLink = icon.parentElement.dataset.tmEditorLink;

		const tooltipString = this.strings.notEnoughCredit.replace(
			'CREDITS_ACTION',
			'document.dispatchEvent(new Event(\'WPMLTooltipCreditsFixClicked\'));'
		);

		return tmEditorLink
			? tooltipString.replace('TRANSLATE_LINK', tmEditorLink)
			: tooltipString
	}

	displayInsufficientBalanceWarning(jobs) {
		this.replaceIcoClassForJobs(
			R.pluck('jobId', jobs),
			['otgs-ico-refresh-spin'],
			'otgs-ico-warning',
			this.notEnoughCreditToolTip.bind(this)
		);
	}

	markJobsAsCancelled(jobs) {
		this.replaceIcoClassForJobs(
			R.pluck('jobId', jobs),
			['otgs-ico-refresh-spin'],
			'otgs-ico-not-translated',
			this.strings.cancelled
		);
	}

	/**
	 * @param {array} jobs
	 */
	setStatusIconToSpin(jobs) {
		const addIconAndTitle = jobId => {
			return {jobId, icon: this.getIcon(jobId), 'title': this.getIconTitle(jobId)}
		};
		const getIconClass = evolve({icon: icon => icon.classList[0]});
		this.previousStates = pluck('jobId', jobs)
			.map(addIconAndTitle)
			.filter(prop('icon'))
			.map(getIconClass);
		this.replaceIcoClassForJobs(
			R.pluck('jobId', jobs),
			['otgs-ico-refresh', 'otgs-ico-in-progress', 'otgs-ico-waiting', 'otgs-ico-edit'],
			'otgs-ico-refresh-spin',
			this.strings.refreshing,
			true
		);
	}

	getPreviousState(jobId, type) {
		const state = this.previousStates.filter(propEq('jobId', jobId));
		return state.length ? state[0][type] : '';
	}

	/**
	 * @param {Array} jobs
	 */
	updateDownloadedJobIcon(jobs) {
		const getIconClass = job => {
			if (doesNeedReview(job)) {
				return 'otgs-ico-needs-review';
			}

			if (this.isInTMDashboard()) {
				return 'otgs-ico-translated';
			}

			return 'otgs-ico-edit';
		};

		jobs.forEach(job => {
			this.replaceIcoClassForJobs(
				[job.jobId],
				['otgs-ico-refresh-spin'],
				getIconClass(job),
				job.label
			);

			this.setJobLink(job);
		});
	}

	revertBack(jobs) {
		pluck('jobId', jobs).forEach(jobId => {
			const icon = this.getIcon(jobId);

			// the icon may not exist when a job is newly created by  Translate Everything process and an original icon was "Not translated"
			if (!icon) {
				return;
			}

			const previousIcon = this.getPreviousState(jobId, 'icon');
			if (previousIcon) {
				icon.classList.replace('otgs-ico-refresh-spin', previousIcon);
			}
			const previousTitle = this.getPreviousState(jobId, 'title');
			if (previousTitle) {
				icon.setAttribute('title', previousTitle);
				initTooltip(icon, 'otgs');
			}
			icon.parentElement.removeEventListener('click', disableLink, true);
		});
	}

	/**
	 * @param {array} jobsIds
	 * @param {array} oldClasses
	 * @param {string} newClass
	 * @param {string|null} newTooltip
	 * @param {boolean} disabled
	 */
	replaceIcoClassForJobs(jobsIds, oldClasses, newClass, newTooltip = null, disabled = false) {
		const icons = this.getJobIcons(jobsIds);
		const updateClasses = R.tap(icon => oldClasses.forEach(oldClass => icon.classList.replace(oldClass, newClass)));
		if (newTooltip) {
			const updateTooltip = icon => {
				const hasAnyOfClasses = R.pipe(R.filter(className => icon.classList.contains(className)), R.length);

				if (hasAnyOfClasses(oldClasses)) {
					const tippyInstance = icon._tippy;
					tippyInstance && tippyInstance.destroy();
					icon.setAttribute('title', typeof newTooltip === 'function' ? newTooltip(icon) : newTooltip);
					initTooltip(icon, 'otgs');
				}
			};
			R.forEach(updateTooltip, icons);
		}
		R.forEach(updateClasses, icons);

		if (disabled) {
			R.map(icon => icon.parentElement.addEventListener('click', disableLink, true), icons);
		} else {
			R.map(icon => icon.parentElement.removeEventListener('click', disableLink, true), icons);
		}
	}

	getJobIcons(jobIds) {
		const query = jobId => `.js-wpml-translate-link[data-tm-job-id="${jobId}"] i`;

		return jobIds.map(query).map(query => this.document.querySelector(query)).filter(R.identity);
	}

	setJobLink(job) {
		const query = `.js-wpml-translate-link[data-tm-job-id="${job.jobId}"]`;
		const aTag = this.document.querySelector(query);
		if (aTag) {
			aTag.setAttribute('href', job.viewLink);
			dispatch('job-link-updated', job);
		}
	}

	/**
	 * @returns {boolean}
	 */
	isInMetaBox() {
		return !!document.getElementById('icl_div') && 'undefined' !== typeof WPMLMetaBox && WPMLMetaBox.refresh;
	}

	/**
	 * @returns {boolean}
	 */
	isInTMDashboard() {
		return !!document.getElementById('icl-tm-translation-dashboard');
	}

	getIcon(jobId) {
		return document.querySelector(`.js-wpml-translate-link[data-tm-job-id="${jobId}"] i`)
	}

	getIconTitle(jobId) {
		const icon = this.getIcon(jobId);
		if (icon) {
			return icon.title ? icon.title : icon.dataset.originalTitle;
		} else {
			return null;
		}
	}
}
