import {evolve, pluck, propEq, prop, identity, forEach} from 'ramda';
import {JOB_STATUSES, doesNeedReview} from "../../jobs/model/jobs";
import {__} from "wpml-common-js-source/src/i18n";

export class TranslationQueue {

	/**
	 * @param {Document} document
	 * @param {Object} strings
	 */
	constructor(document, strings) {
		this.document = document;
		this.strings = strings;
		this.previousStates = [];
		this.getIcon = this.getIcon.bind(this);
	}

	/**
	 * @param {Array} jobs
	 */
	setStatusIconToSpin(jobs) {
		const addIcon = jobId => {return {jobId, icon: this.getIcon(jobId)}};
		const getIconClass = evolve({icon: icon => icon.classList[0]});
		this.previousStates = pluck('jobId', jobs)
			.map(addIcon)
			.filter(prop('icon'))
			.map(getIconClass);
		this.replaceIcoClassForJobs(pluck('jobId', jobs), 'otgs-ico-waiting', 'otgs-ico-refresh-spin');
		this.replaceIcoClassForJobs(pluck('jobId', jobs), 'otgs-ico-in-progress', 'otgs-ico-refresh-spin');
	}

	getPreviousState(jobId) {
		const state = this.previousStates.filter(propEq('jobId', jobId));
		return state.length ? state[0]['icon'] : '';
	}

	updateDownloadedJobIcon(jobs) {
		jobs.forEach((job) => {
			const row = this.document.querySelector(`.js-wpml-job-row[data-job-id="${job.jobId}"]`);

			if (!row) {
				return;
			}

			const statusText = row.querySelector('.column-status span');
			if (job.status == JOB_STATUSES.COMPLETED) {
				statusText.innerHTML = `<i class="otgs-ico-translated"></i>${__('Complete', 'wpml-translation-management')}`;
			} else if (doesNeedReview(job)) {
				statusText.innerHTML = `<i class="otgs-ico-needs-review"></i>${__('Pending review', 'wpml-translation-management')}`;
			}
		});
	}

	displayInsufficientBalanceWarning(jobs) {
		this.replaceIcoClassForJobs(
			pluck('jobId', jobs),
			'otgs-ico-refresh-spin',
			'otgs-ico-warning',
			__('Insufficient funds', 'wpml-translation-management')
		);
	}

	markJobsAsCancelled(jobs) {
		this.replaceIcoClassForJobs(
			pluck('jobId', jobs),
			'otgs-ico-refresh-spin',
			'otgs-ico-not-translated',
			this.strings.cancelled
		);
	}

	revertBack(jobs) {
		pluck('jobId', jobs).forEach(jobId=> {
			const previous = this.getPreviousState(jobId);
			if (previous) {
				const icon = this.getIcon(jobId);
				icon.classList.replace('otgs-ico-refresh-spin', previous);
			}
		});
	}

	replaceIcoClassForJobs(jobIds, oldClass, newClass, newStatus = null) {
		const updateClass = icon => icon.classList.replace(oldClass, newClass);

		const setStatus = icon => icon.parentNode.childNodes[1].nodeValue = newStatus;
		const updateStatus = newStatus ? setStatus : identity;

		const icons = jobIds.map(this.getIcon).filter(identity);
		forEach(updateClass, icons);
		forEach(updateStatus, icons);
	}

	getIcon(jobId) {
		return this.document.querySelector(`.js-wpml-job-row[data-job-id="${jobId}"] .column-status i`)
	}

}
