import React from 'react';

import {SettingsConsumer} from '../context/SettingsProvider';
import {SelectionConsumer} from '../context/SelectionProvider';
import {JobsConsumer} from '../context/JobsProvider';
import connect from '../context/connect';
import {__} from "wpml-common-js-source/src/i18n";

import TitleCell from './ListItem/Cells/Title';
import LanguageCell from './ListItem/Cells/Language';
import RegularCell from './ListItem/Cells/Regular';
import TranslatorCell from './ListItem/Cells/Translator';
import StatusCell from './ListItem/Cells/Status';

import BatchSyncAction from './ListItem/Actions/BatchSync';
import DownloadTranslationsAction from './ListItem/Actions/DownloadTranslations';
import DownloadXLIFFAction from './ListItem/Actions/DownloadXLIFF';
import CancelAction from './ListItem/Actions/Cancel';
import {isLocal, JOB_STATUSES} from '../model/jobs';
import initTooltip from "../util/tooltip";

export default class ListItem extends React.Component {
	constructor(props) {
		super(props);
		this.tooltip = React.createRef();
	}

	componentDidMount() {
		this.refreshTooltip();
	}

	componentDidUpdate() {
		this.refreshTooltip();
	}

	refreshTooltip() {
		initTooltip(this.tooltip.current);
	}

	isSelectable(job) {
		if (isLocal(job)) {
			return job.status === JOB_STATUSES.WAITING_FOR_TRANSLATOR || job.status === JOB_STATUSES.IN_PROGRESS;
		} else {
			return job.status === JOB_STATUSES.TRANSLATION_READY_TO_DOWNLOAD || job.status === JOB_STATUSES.COMPLETED;
		}
	}

	render() {
		const {
				  job,
				  columns = {},
				  isJobSelected,
				  selectJob,
				  jobsStatuses,
				  progress,
				  hasExternalActions,
				  translationService,
			  } = this.props;

		const isInProgress = job => progress && progress.jobs && progress.jobs.find(inProgressJob => job.id === inProgressJob.id && job.type === inProgressJob.type) !== undefined;

		const actions = [];

		if (translationService) {
			actions.push(<BatchSyncAction key="batch-sync-action" job={job}/>);
			actions.push(<DownloadTranslationsAction key="translation-download-action" job={job}/>);
			actions.push(<DownloadXLIFFAction key="download-xliff-action" job={job}/>);
		}

		actions.push(<CancelAction key="cancel-action" job={job}/>);

		if (isInProgress(job)) {
			actions.push(<span key="spinner" className="spinner is-active"/>);
		}

		const externalActions = [];
		if (job.ts_status && job.ts_status.links.length) {
			job.ts_status.links.forEach(({url, label}) => {
				externalActions.push(<a key={`ts-link-${label}`} className="otgs-external-link wpml-external-link" rel="parent"
										target="_blank" href={url}>{label}</a>);
			});
		}

		return <tr>
			<td>
				{
					this.isSelectable(job) && (
						<input
							type="checkbox"
							checked={isJobSelected(job)}
							onChange={() => selectJob(job, !isJobSelected(job))}
							title={`Select job ${job.id} - ${job.element.name}`}
						/>
					)
				}
			</td>
			{
				Object.keys(columns).map((index) => {
					switch (index) {
						case 'title':
							return <TitleCell key={`${job.id}-title`} {...job.element} />;
						case 'languages':
							return <LanguageCell key={`${job.id}-languages`} {...job.languages} />;
						case 'translator' :
							return <TranslatorCell key={index} job={job}/>;
						case 'status':
							return <StatusCell key={index} job={job} jobsStatuses={jobsStatuses}/>;
						case 'batch_name':
							return <RegularCell key={index} text={job.batch.name}/>;
						case 'id':
							if ( job.extra_data && job.extra_data.editor_job_id ) {
								return <RegularCell key={index} text={
									<span
										className="js-otgs-popover-tooltip"
										title={`${__('Editor Job ID', 'wpml-translation-management' )}: ${job.extra_data.editor_job_id}`}
										ref={this.tooltip}
									>
										{ job[ index ] }
									</span>
								} />;
							}
							return <RegularCell key={index} text={job[index]}/>;
						default:
							return <RegularCell key={index} text={job[index]}/>;
					}
				})
			}
			<td className="column-actions" >{actions}</td>
			{
				hasExternalActions() && <td className="column-external-actions">{externalActions}</td>
			}
		</tr>;
	}

}

export const connectedListItem = connect(ListItem, SettingsConsumer, JobsConsumer, SelectionConsumer);
