import AMSActivationStatuses from '../../ate/Models/AMSActivationStatuses';
import WPMLTMSettings from '../../shared/WPMLTMSettings';
import ATEControlsUI from "./ATEControlsUI";
import * as R from 'ramda';

class ATEControls extends WPMLTMSettings {
	/**
	 *
	 * @param {AMSAPI} amsAPI
	 * @param {object} statusOnly
	 */
	constructor(amsAPI, {preventReload = false} = {}) {
		super();

		this.visible = false;
		this.ui      = new ATEControlsUI({preventReload});

		this.amsAPI      = amsAPI;
		this.statusTypes = Object.keys(this.ateSettings.statuses).map((key) => this.ateSettings.statuses[key].type);
	}

	init() {
		this.initUI();
		this.checkActivation();
		this.initEventHandlers();
	}

	initUI() {
		this.ui.hideStatus();
		this.ui.hideErrors();
	}

	show() {
		this.visibility(true);
	}

	hide() {
		this.visibility(false);
	}

	visibility(show) {
		this.visible = show;
		this.ui.setControlsVisibility(show);
	}

	updateControlsStatus() {
		if (this.visible) {
			if (this.ATERequiresActivation()) {
				this.enableATE();
			} else {
				if (this.ATEIsActive()) {
					this.initEventHandlers();
				}
				this.updateStatusMessage();
			}
		}
		this.initEventHandlers();
	}

	updateStatusMessage() {
		if(this.ateSettings.currentStatus !== AMSActivationStatuses.active) {
			this.ui.showAteControls();
			this.ui.showStatus({status: this.getStatus(), statusTypes: this.statusTypes});
		} else {
			this.ui.hideAteControls();
		}
	}

	getStatus(ateStatus = null) {
		if (null === ateStatus) {
			ateStatus = this.ateSettings.currentStatus;
		}
		return this.ateSettings.statuses[ateStatus];
	}

	ATERequiresActivation() {
		return this.ateSettings.currentStatus === AMSActivationStatuses.nonActive;
	}

	ATEIsActive() {
		return this.ateSettings.currentStatus === AMSActivationStatuses.active;
	}

	initEventHandlers() {
		this.ui.initEventHandlers();
	}

	checkActivation() {
		this.ui.clearCache();
		if (this.ateSettings.currentStatus === AMSActivationStatuses.enabled) {
			this.ui.updateErrors();
			this.ui.showSpinner();
			this.amsAPI.checkStatus()
				.then(({activated} = {activated: false}) => {
					if (activated) {
						this.ateSettings.currentStatus = AMSActivationStatuses.active;
						this.handleSynchronize();
						// return this.updateConsoleURL();
					}
				})
				.finally(() => {
					this.updateControlsStatus();
					this.ui.updateActions(true);
					this.ui.hideSpinner();
				});
		} else {
			this.updateControlsStatus();
			// return this.updateConsoleURL();
		}
	}

	enableATE(successCallback,failureCallback) {
		if (this.enablingATE) {
			return;
		}

		this.enablingATE = true;
		let hasError = false;

		this.ui.clearCache();
		this.ui.updateErrors();
		this.ui.showSpinner();

		this.visibility(true);
		this.updateControlsStatus();

		this.amsAPI.registerClient()
			.then(response => {
				if (!response.enabled) {
					throw R.propOr([], 'error', response);
				}

				return response;
			})
			.then(() => {
				this.ateSettings.currentStatus = AMSActivationStatuses.enabled;
				this.ui.clearCache();
				this.updateControlsStatus();
				this.updateConsoleURL();
				if (successCallback) {
					successCallback();
				}
			})
			.catch(reason => {
				hasError = true;
				this.ui.updateErrors(reason);
				if (failureCallback) {
					failureCallback();
				}
			})
			.finally(() => {
				this.ui.hideSpinner();
				this.enablingATE = false;

				if ( ! hasError ) {
					this.checkActivation();
					this.ui.updateActions();
				}
			});
	}

	async enableATEWithoutUI() {
		return this.amsAPI.registerClient()
			.then( response => {
				if ( response.enabled ) {
					this.ateSettings.currentStatus = AMSActivationStatuses.enabled;
				}

				return response;
			})

	}

	updateConsoleURL() {
		return this.amsAPI.updateConsoleURL()
			.catch((reason) => {
				this.ui.updateErrors(reason);
			})
			.then((url) => {
				this.ui.updateAteConsoleLink(url);
			});
	}

	handleSynchronize() {
		return this.amsAPI.synchronize();
	}

	isStatusOnly() {
		return this.ui.isStatusOnly();
	}

}

export default ATEControls;
