import WPMLTMSettings from '../shared/WPMLTMSettings';
import {createHooks} from '@wordpress/hooks';

export const hooks = createHooks();

class Request extends WPMLTMSettings {
	/**
	 * @param {string} url
	 * @return {Promise}
	 */
	get(url) {
		return this.request(url, 'GET');
	}

	/**
	 * @param {string} url
	 * @param {string|object} body
	 * @return {Promise}
	 */
	post(url, body = null) {
		return this.request(url, 'POST', body);
	}

	async download(url) {
		return fetch(url, this.generateArgs('GET')).then(response => {
			if (!response.ok) {
				return this.buildError(response);
			}
			return response.blob();
		});
	}

	/**
	 * @param {string} url
	 * @param {string} verb
	 * @param {string|object} body
	 * @return {Promise}
	 */
	request(url, verb, body = null) {

		const args = this.generateArgs(verb, body);

		if (this.restUrl.includes('?')) {
			url = url.replace('?', '&');
		}

		return this.handleRequest(this.restUrl + url, args);
	}

	/**
	 * @param {string} url
	 * @param {object} args
	 * @return {Promise<Response>}
	 */
	handleRequest(url, args) {
		return fetch(url, args)
			.then(response => {
				if (!response.ok) {
					return this.buildError(response);
				}
				return response.json();
			});
	}

	generateArgs(verb, body = null) {
		const args = {
			method: verb.toUpperCase(),
			headers: {
				'Accept': 'application/json',
				'Content-Type': 'application/json',
				'X-WP-Nonce': this.restNonce,
			},
			credentials: 'same-origin',
		};

		if (args.method !== 'GET' && body) {
			if (typeof body === 'string') {
				args.body = body;
			} else {
				args.body = JSON.stringify(body);
			}
		}

		return args;
	};

	buildError(response) {
		return response.json().then(json => {
			const error = new Error(json.message || response.statusText)
			return Promise.reject(Object.assign(error, {response}))
		});
	};
}

export default Request;