import React, {useEffect, useState} from 'react';
import TranslationMethod from '../../../../../../../src/js/components/translation/TranslationMethod';
import {useStoreActions, useStoreState} from 'easy-peasy';
import {identity, curryN, __ as _, propOr, includes} from 'ramda';
import Spin from 'antd/lib/spin';
import 'antd/lib/spin/style/index.css';
import Modal from "antd/lib/modal";
import '../style/index.scss';
import {__} from 'wpml-common-js-source/src/i18n';
import {createInterpolateElement} from '@wordpress/element';
import Row from "antd/lib/row";
import Col from "antd/lib/col";
import "antd/lib/grid/style/index.css";
import Radio from "antd/lib/radio";
import Space from "antd/lib/space";
import 'antd/lib/space/style/index.css';
import Button from "../../../../../../../src/js/shared/components/button";
import PostCount from "./PostCount";
import useSaving from "../../../../../../../src/js/shared/useSaving";
import {dispatch} from "../../../../../../../src/js/shared/event";
import {parse} from "query-string";
import {callEndpoint} from "wpml-common-js-source/src/Ajax";
import ModalCloseIcon from '../../../../../../../src/icons/ModalCloseIcon.svg';

export default () => {
	const {
		method,
		reviewMode,
		saving,
		confirmingTranslateEverything,
		hasTranslationService,
		isClassicEditor,
		endpoints,
		urls,
		hasJobsInProgress,
		disableTranslateEverything
	} = useStoreState(identity);

	const {setMethod, setReviewMode, updateMethod} = useStoreActions(identity);

	useEffect(() => {
		dispatch('auto-translation-method', method);
	}, [method]);

	const save = async (content) => {
		updateMethod('automatic');
		await callEndpoint(endpoints.setTranslateEverything, {
			method: 'automatic',
			translateEverything: true,
			onlyNew: content === 'new'
		});
	};

	const saveIfATESubscribed = () => {
		const content = propOr(false, 'content', parse(location.search));
		const nonce = propOr(false, 'nonce', parse(location.search));
		if (includes(content, ['new', 'existing']) && nonce === wpml_translation_method.nonce) {
			save(content);
		}
	};

	useEffect(saveIfATESubscribed, []);

	const reasonText = isClassicEditor
		? __("You must use the <s>Advanced Translation Editor</s> if you want to translate your content automatically.", 'wpml-translation-management')
		: __("You currently use a translation service to translate your content. To translate your site automatically, first disconnect your translation service in your <a><s>Translation Management settings</s></a>.", 'wpml-translation-management');

	const reason = createInterpolateElement(
		reasonText,
		{
			's': <strong/>,
			'a': <a href={wpml_translation_method.translatorsTabLink}/>
		}
	);

	return (
		<div>
			<TranslationMethod
				isWizard={false}
				method={method}
				setMethod={setMethod}
				reviewMode={reviewMode}
				setReviewMode={setReviewMode}
				allowTranslateEverything={!hasTranslationService && !isClassicEditor}
				reason={reason}
				tmDashboardUrl={urls.tmDashboard}
				hasJobsInProgress={hasJobsInProgress}
				forceShowNotice={disableTranslateEverything}
			/>
			{saving && <Spin spinning={true}/>}
			{confirmingTranslateEverything && <Confirm/>}
		</div>
	);
};

const Confirm = () => {
	const {hasSubscription, endpoints, postTypes} = useStoreState(identity);
	const {setMethod, setConfirming} = useStoreActions(identity);
	const [content, setContent] = useState('new');
	const [isSaving, save, error, clearError] = useSaving(endpoints.setTranslateEverything);

	const onCancel = () => {
		setConfirming(false);
	};

	const signUpOnATETab = () => {
		const returnURL = window.location.href + '&content=' + content + '&nonce=' + wpml_translation_method.nonce;
		window.location.href = wpml_translation_method.createAccountLink + '&return_after_pay_url=' + encodeURIComponent(returnURL);
	};

	const onStartTranslating = () => {
		if (hasSubscription) {
			save({
					method: 'automatic',
					translateEverything: true,
					onlyNew: content === 'new'
				},
				() => setConfirming(false)
			);
		} else {
			signUpOnATETab();
		}
	};

	const withStrong = curryN(2, createInterpolateElement)(_, {'s': <strong/>});

	const yes = withStrong(
		hasSubscription
			? __('Yes, automatically translate all of my <s>existing posts and pages</s> right now, plus <s>new content</s> as I create it.', 'wpml-translation-management')
			: __('Automatically translate all of my <s>existing content</s> right now, plus <s>new content</s> as I create it.', 'wpml-translation-management')
	);

	const no = withStrong(
		hasSubscription
			? __("No, only automatically translate <s>new content</s>.", 'wpml-translation-management')
			: __("Only automatically translate <s>new content</s>.", 'wpml-translation-management')
	);

	const heading = hasSubscription
		? __('Should WPML automatically translate your existing content?', 'wpml-translation-management')
		: __('Which content do you want to translate automatically?', 'wpml-translation-management');

	const translate = hasSubscription
		? __('Start translating automatically', 'wpml-translation-management')
		: __('Create an account to start translating', 'wpml-translation-management');

	return (
		<Modal
			className={'wpml-confirm-translation-method'}
			visible={true}
			onCancel={onCancel}
			footer={null}
			closeIcon={<ModalCloseIcon />}
			width={635}
		>
			<Row>
				<Col>
					<Space direction={'vertical'} size={24} style={{width: '100%'}}>
						{hasSubscription && <p className={'intro-subscription'}>{__(
							'WPML will use credits from your automatic translation account to translate your site immediately as you publish new content or update existing content.',
							'wpml-translation-management'
						)}</p>}
						<h1 className={'heading'}>
							{heading}
						</h1>
						<Radio style={{whiteSpace: 'pre-wrap'}} className={'radio ant-multiple-rows'} value={'existing'} onChange={() => setContent('existing')}
							   checked={content === 'existing'}>
							{yes}
						</Radio>
						<PostCount postTypes={postTypes}/>
						<Radio className={'radio'} value={'new'} onChange={() => setContent('new')}
							   checked={content === 'new'}>
							{no}
							<br /><span className={'small-gray'}>{__('You can translate existing content from the Translation Management Dashboard.', 'wpml-translation-management')}</span>
						</Radio>
						<Row className={'ant-modal-body__footer'}>
							<Col span={8}>
								<Button type={'secondary'} onClick={onCancel}>
									{__('Cancel', 'wpml-translation-management')}
								</Button>
							</Col>
							<Col span={4}>
								<Spin spinning={isSaving}/>
							</Col>
							<Col span={12}>
								<Button type={'primary'} onClick={onStartTranslating} disabled={isSaving || !content}>
									{translate}
								</Button>
							</Col>
						</Row>
					</Space>
				</Col>
			</Row>
		</Modal>
	);
};
