import {action, computed, createStore, thunk} from 'easy-peasy';
import {find, flatten, map, pipe, prop, propEq} from 'ramda';
import Either from 'crocks/Either';
import {callEndpoint} from "wpml-common-js-source/src/Ajax";

const findActiveService = pipe(map(prop('services')), flatten, find(propEq('active', 'active')));

const updateService = (sections, serviceId, newData) => {
    return sections.map(section => {
        const services = section.services.map(service => service.id === serviceId ? {...service, ...newData} : service);

        return {...section, services};
    });
};

const model = (sections, endpoints, logoPlaceholder, isTMTab, show, settingsUrl) => {
    const activeService = findActiveService(sections);

    return {
        show,
        sections,
        logoPlaceholder,
        activeService,
	    settingsUrl,
        selectedService: activeService,
        apiToken: undefined,
	    authModalOpen: false,

	    setAuthModalOpen: action((state, open) => {
	    	state.authModalOpen = open;
	    }),

        selectService: thunk(async (actions, service) => {
            if (isTMTab) {
                const {data} = await callEndpoint(endpoints.selectService, {service_id: service.id});

                if (data.success) {
                    window.location.reload();
                }
            } else {
                actions.serviceSelected(service);
            }
        }),

        serviceSelected: action((state, service) => {
            state.selectedService = service;
	        state.authModalOpen = true;
        }),

        unselectService: action(state => {
        	state.selectedService = undefined;
        	state.authModalOpen = false;
        }),

        setApiToken: action((state, apiToken) => {
           state.apiToken = apiToken;
        }),

        activateService: thunk(async (actions, {service, apiToken}) => {
                const {data} = await callEndpoint(endpoints.activateService, {
                    service_id: service.id, api_token: apiToken
                });

                if (data.success) {
                    actions.serviceActivated(service);
                    return Either.Right(data.data);
                } else {
                    return Either.Left(data.data);
                }
        }),

        serviceActivated: action((state, service) => {
            state.sections = updateService(state.sections, service.id, {active: 'active'});
            service.active = 'active';
            state.activeService = service;
        }),

        deactivateService: thunk(async (actions, serviceId) => {
            const {data} = await callEndpoint(endpoints.deactivateService, {service_id: serviceId});

            if (data.success) {
                actions.serviceDeactivated(serviceId);
                return Either.Right(data.data);
            } else {
                return Either.Left(data.data);
            }
        }),

        serviceDeactivated: action((state, serviceId) => {
            state.sections = updateService(state.sections, serviceId, {active: 'inactive'});
            state.selectedService = undefined;
            state.activeService = undefined;
        }),
    };
};

export default (servicesList, endpoints, logoPlaceholder, isTMTab = false, show = true, settingsUrl = '') => createStore(model(servicesList, endpoints, logoPlaceholder, isTMTab, show, settingsUrl));
